// Copyright 2014 Google Inc. All Rights Reserved.

#include "VideoSink.h"

void VideoSink::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    // This version only supports one codec type. If we ever support another video
    // codec, we can actually have it passed in.
    Service* service = sdr->add_services();
    service->set_id(id());
    MediaSinkService* mss = service->mutable_media_sink_service();
    mss->set_available_type(mCodecType);

    for (vector<shared_ptr<VideoConfiguration> >::iterator it = mVideoConfigurations.begin();
            it != mVideoConfigurations.end(); ++it) {
        shared_ptr<VideoConfiguration> cur = *it;
        VideoConfiguration* vc = mss->add_video_configs();
        vc->set_codec_resolution(cur->codec_resolution());
        vc->set_frame_rate(cur->frame_rate());
        vc->set_width_margin(cur->width_margin());
        vc->set_height_margin(cur->height_margin());
        vc->set_density(cur->density());
        vc->set_decoder_additional_depth(cur->decoder_additional_depth());
        if (mViewingDistance != 0) {
            vc->set_viewing_distance(mViewingDistance);
        }
        vc->set_pixel_aspect_ratio_e4(cur->pixel_aspect_ratio_e4());
    }
}

void VideoSink::handleDataAvailable(uint64_t timestamp,
        const shared_ptr<IoBuffer>& data, size_t offset) {
    uint8_t* ptr = (uint8_t*) data->raw() + offset;
    size_t len = data->size() - offset;
    mCallbacks->dataAvailableCallback(mCurrentSessionId, timestamp, data, ptr, len);
}

uint32_t VideoSink::addSupportedConfiguration(int codecResolution, int fps, int32_t widthMargin,
        int32_t heightMargin, int32_t density, int32_t decoderAdditionalDepth,
        int32_t pixelAspectRatioE4) {
    shared_ptr<VideoConfiguration> vc(new VideoConfiguration);
    vc->set_codec_resolution((VideoCodecResolutionType) codecResolution);
    vc->set_frame_rate(
            (fps == 60) ? VIDEO_FPS_60 : VIDEO_FPS_30);
    vc->set_width_margin(widthMargin);
    vc->set_height_margin(heightMargin);
    vc->set_density(density);
    vc->set_decoder_additional_depth(decoderAdditionalDepth);
    vc->set_pixel_aspect_ratio_e4(pixelAspectRatioE4);
    mVideoConfigurations.push_back(vc);
    return mVideoConfigurations.size() - 1;
}

int VideoSink::handleSetup(int mediaCodecType) {
    int status = STATUS_SUCCESS;
    if (mediaCodecType != MEDIA_CODEC_VIDEO_H264_BP) {
        return STATUS_MEDIA_CONFIG_MISMATCH;
    }

    mCallbacks->setupCallback(mediaCodecType);

    sendConfig(Config_Status_STATUS_READY);

    if (mAutoStartProjection) {
        setVideoFocus(VIDEO_FOCUS_PROJECTED, true);
    }
    return status;
}

int VideoSink::handleCodecConfig(void* config, size_t len) {
    return mCallbacks->codecConfigCallback((uint8_t*)config, len);
}

int VideoSink::codecResolutionToPixels(const VideoConfiguration& config,
        int32_t* width, int32_t* height) {
    // This function is temporary till we get service discovery working. After that we just need
    // to send up an index and have the sink deal with it.
    int status = STATUS_SUCCESS;
    switch(config.codec_resolution()) {
        case VIDEO_800x480:
            *width = 800;
            *height = 480;
            break;
        case VIDEO_1280x720:
            *width = 1280;
            *height = 720;
            break;
        case VIDEO_1920x1080:
            *width = 1920;
            *height = 1080;
            break;
        default:
            status = STATUS_MEDIA_CONFIG_MISMATCH;
    }
    return status;
}

int VideoSink::handleMediaConfiguration(int index) {
    mCallbacks->sourceVideoConfigCallback(index);
    return STATUS_SUCCESS;
}

void VideoSink::playbackStart(int32_t sessionId) {
    mCallbacks->playbackStartCallback(sessionId);
}

void VideoSink::playbackStop(int32_t sessionId) {
    mCallbacks->playbackStopCallback(sessionId);
}

bool VideoSink::handleVideoFocusRequest(const VideoFocusRequestNotification& req) {
    int reason = UNKNOWN;
    if (req.has_reason()) {
        reason = req.reason();
    }
    mCallbacks->videoFocusCallback(req.mode(), reason);
    return true;
}

void VideoSink::setVideoFocus(int focusMode, bool unsolicited) {
    VideoFocusNotification vfn;
    vfn.set_focus((VideoFocusMode) focusMode);
    vfn.set_unsolicited(unsolicited);
    IoBuffer buf;
    mRouter->marshallProto(MEDIA_MESSAGE_VIDEO_FOCUS_NOTIFICATION, vfn, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

unsigned int VideoSink::getNumberOfConfigurations() {
    return mVideoConfigurations.size();
}
